<?php

/**
 * Custom Status class that handles loading, adding, editing, and deleting the statuses
 */
class WC_Custom_Status {

    var $term_id, $term_taxonomy_id, $status_name, $status_icon, $status_color, $sends_email, $email_to, $custom_email_address,
        $from_address, $from_name, $email_subject, $display_essential_info, $display_custom_info,
        $custom_title, $custom_info, $payment_status;

    public static $default_status_list = array('pending','failed','processing','completed','on-hold','cancelled','refunded');

    /**
     * Deletes the custom status from the custom table and the terms table
     * @param  boolean $alt_term [can be integer, to handle the multi and the single deletion]
     * @return [boolean]            [whether the deletion was successful]
     */
    function delete( $alt_term = false ) {
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";

        $alt_term = empty($alt_term)? $_POST['alt_term']:$alt_term;

        // Get the orders with the - to delete - order status as term
        $orders = get_objects_in_term( $this->term_id, 'shop_order_status' );
        if( is_wp_error( $orders ) ) {
            return false;
        }

        // Place the orders that have this status with the alternative
        foreach( $orders as $o ) {
            $z = wp_set_object_terms( intval($o), intval($alt_term), 'shop_order_status', false );
        }


        // Delete the term
        $delete_term = wp_delete_term($this->term_id, 'shop_order_status');


        // Delete the record from the custom table
        $delete_custom = $wpdb->delete( $table_name, array('term_id'=>$this->term_id) );

        return $delete_term && $delete_custom;
    }

    /**
     * Adds/Updates the status
     * @param [array]  $post_fields   [submitted data]
     * @param boolean $update_status [whether to update the status or not]
     */
    function add_status($post_fields, $update_status = false){
        try {
            if(isset($_REQUEST['status_id'])){
                $status_id = (int)strip_tags($_REQUEST['status_id']);
            }
            // Check whether any of the required fields are missing
            WC_Custom_Status_Plugin::check_missing_fields(array('status_name', 'status_icon', 'action_icon', 'display_for'), $post_fields);
            // Set the object to the submitted data
            $status_name = sanitize_title($post_fields['status_name']);
            $this->status_name = $status_name;
            $this->status_icon = $post_fields['status_icon'];
            $this->status_color = $post_fields['status_color'];
            $this->status_color_type = $post_fields['status_color_type'];
            $this->action_icon = $post_fields['action_icon'];
            $this->display_for = implode(',',$post_fields['display_for']);
            // If we're not updating and there exists a status with the same name throw an exception
            if(!$update_status && $this->is_valid_status($this->status_name)){
                throw new Exception("Status {$this->status_name} already exists! Please choose another name");
            }
            // If we're to send an email
            if(isset($post_fields['sends_email']) && $post_fields['sends_email'] == 'yes'){
                $this->sends_email = '1';

                // Check required fields for sending the email are submitted
                WC_Custom_Status_Plugin::check_missing_fields(array('from_address','from_name','email_subject','email_to', 'custom_title'),$post_fields);
                // Set the status to contain email options
                $this->from_address = sanitize_text_field($post_fields['from_address']);
                $this->from_name = sanitize_text_field($post_fields['from_name']);
                $this->email_subject = sanitize_text_field($post_fields['email_subject']);
                $this->email_to = sanitize_text_field($post_fields['email_to']);
                $this->custom_title = $post_fields['custom_title'];

                // Display order information?
                if(isset($post_fields['display_essential_info']) && $post_fields['display_essential_info'] == 'yes'){
                    $this->display_essential_info = '1';
                } else {
                    $this->display_essential_info = '0';
                }

                if(isset($post_fields['email_to']) && $post_fields['email_to'] == 'custom'){
                    WC_Custom_Status_Plugin::check_missing_fields(array('custom_email_address'),$post_fields);
                    $this->custom_email_address = sanitize_text_field($post_fields['custom_email_address']);
                }

                // Display custom message?
                if(isset($post_fields['display_custom_info']) && $post_fields['display_custom_info'] == 'yes'){
                    $this->display_custom_info = '1';
                    // Check whether the custom message or submitted or not
                    WC_Custom_Status_Plugin::check_missing_fields(array('custom_info'),$post_fields);
                    $this->custom_info = $post_fields['custom_info'];
                } else {
                    $this->display_custom_info = '0';
                }
            } else {
                $this->sends_email = '0';
            }

            if(isset($post_fields['display_completed']) && $post_fields['display_completed'] == 'yes'){
                $this->display_completed = '1';
            } else {
                $this->display_completed = '0';
            }

            if(isset($post_fields['display_processing']) && $post_fields['display_processing'] == 'yes'){
                $this->display_processing = '1';
            } else {
                $this->display_processing = '0';
            }

            if(isset($post_fields['can_cancel']) && $post_fields['can_cancel'] == 'yes'){
                $this->can_cancel = '1';
            } else {
                $this->can_cancel = '0';
            }
            if(isset($post_fields['display_in_reports']) && $post_fields['display_in_reports'] == 'yes'){
                $this->display_in_reports = '1';
            } else {
                $this->display_in_reports = '0';
            }

            // If we're not updating create a new term
            if(!$update_status){
                $term_details = wp_insert_term($this->status_name, 'shop_order_status');
            }
            if( (!empty($term_details) && is_array($term_details)) || $update_status){
                if(!$update_status){
                    $this->term_id = $term_details['term_id'];
                    $this->term_taxonomy_id = $term_details['term_taxonomy_id'];
                }
                // Make the data ready for insertion/update in the database
                global $wpdb;
                $table_name = $wpdb->prefix . "woocommerce_order_status_action";
                $data = array('term_id' => (!empty($this->term_id) ? $this->term_id : (int)strip_tags($post_fields["term_id"])), 'term_taxonomy_id' => (!empty($this->term_taxonomy_id) ? $this->term_taxonomy_id : (int)strip_tags($post_fields["term_taxonomy_id"])),
                    'status_name' => sanitize_title($this->status_name), 'status_icon' => $this->status_icon, 'status_color' => $this->status_color,
                    'sends_email' => $this->sends_email, 'email_to' => $this->email_to, 'custom_email_address' => $this->custom_email_address,
                    'from_address' => $this->from_address, 'from_name' => $this->from_name,
                    'email_subject' => $this->email_subject, 'display_essential_info' => $this->display_essential_info,
                    'display_custom_info' => $this->display_custom_info, 'custom_title' => $this->custom_title,
                    'custom_info' => $this->custom_info, 'payment_status' => '0',
                    'action_icon' => $this->action_icon, 'new_status' => sanitize_title($this->status_name), 'display_for' => $this->display_for,
                    'created' => date("Y-m-d H:i:s"),'updated' => date("Y-m-d H:i:s"),
                    'display_completed' => $this->display_completed, 'display_processing' => $this->display_processing, 'status_color_type' => $this->status_color_type,
                    'can_cancel' => $this->can_cancel,
                    'display_in_reports' => $this->display_in_reports
                );
                if(!$update_status){
                    if($wpdb->insert($table_name, $data, array('%d','%d','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s'))){
                        return true;
                    } else {
                        $wpdb->print_error();
                        return false;
                    }
                } else {
                    if($wpdb->update($table_name, $data, array('id' => $status_id),array('%d','%d','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s'),array('%d'))){
                        return true;
                    } else {
                        $wpdb->print_error();
                        return false;
                    }
                }
            } else {
                throw new WPErrorException($term_details->get_error_messages());
            }

        } catch(RequiredFieldMissingException $e) {
            throw $e;
        }
    }

    /**
     * If an order was set to our status, sends the mail.
     * @param  [integer] $order_id   [order id]
     * @param  [string] $old_status [old status]
     * @param  [string] $new_status [new status]
     * @return boolean
     */
    function process_status($order_id,$old_status,$new_status){
        $this->load_status_from_db($new_status);
        // If the status is configured to send an email
        if($this->sends_email == '1'){
            // Are we going to include order info, message?
            if($this->display_custom_info != '1'){
                $this->custom_info = '';
            }
            if($this->display_essential_info == '1'){
                $display_essential_info = true;
            } else {
                $display_essential_info = false;
            }
            // Send the email to specified users
            $wc_custom_email = new WC_Custom_Status_Email($new_status,$this->custom_title,$this->email_subject,$this->from_name, $this->from_address, $this->custom_info, $display_essential_info, $this->custom_email_address);
            switch($this->email_to){
                case 'customer':
                    $wc_custom_email->trigger($order_id,'customer');
                    break;
                case 'admin':
                    $wc_custom_email->trigger($order_id,'admin');
                    break;
                case 'both':
                    $wc_custom_email->trigger($order_id,'customer');
                    $wc_custom_email->trigger($order_id,'admin');
                    break;
                case 'custom':
                    $wc_custom_email->trigger($order_id,'custom');
                    break;
            }
        }

        return true;
    }

    /**
     * Load the status from the database and set the current object's data
     * @param  [string] $status_name [status/term name]
     * @return boolean
     */
    function load_status_from_db($status_name){
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";
        $query = "SELECT * FROM $table_name WHERE status_name = '$status_name'";
        $status_info = $wpdb->get_row($query,ARRAY_A);
        if($status_info != null){
            $this->status_name = $status_name;
            $this->status_icon = $status_info['status_icon'];
            $this->status_color = $status_info['status_color'];
            $this->term_id = $status_info['term_id'];
            $this->term_taxonomy_id = $status_info['term_taxonomy_id'];
            $this->sends_email = $status_info['sends_email'];
            $this->email_to = $status_info['email_to'];
            $this->custom_email_address = $status_info['custom_email_address'];
            $this->email_subject = $status_info['email_subject'];
            $this->from_name = $status_info['from_name'];
            $this->from_address = $status_info['from_address'];
            $this->display_essential_info = $status_info['display_essential_info'];
            $this->display_custom_info = $status_info['display_custom_info'];
            $this->custom_title = $status_info['custom_title'];
            $this->custom_info = $status_info['custom_info'];
            $this->new_status = $status_info['new_status'];
            $this->action_icon = $status_info['action_icon'];
            $this->display_for = explode(',',$status_info['display_for']);
            $this->display_completed = $status_info['display_completed'];
            $this->display_processing = $status_info['display_processing'];
            $this->status_color_type = $status_info['status_color_type'];
            $this->can_cancel = $status_info['can_cancel'];
            $this->display_in_reports = $status_info['display_in_reports'];
            return true;
        } else {
            return false;
        }
    }

    /**
     * A valid status is checked on creation. IT simply checks whether a status already exists with the same name.
     * @param  [string]  $status_name [status name]
     * @return boolean
     */
    static function is_valid_status($status_name){
        if(in_array($status_name,WC_Custom_Status::$default_status_list)){
            return true;
        }
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";
        $query = "SELECT status_icon FROM $table_name WHERE status_name = '$status_name'";
        $result = $wpdb->get_row($query);
        if(!$result){
            return false;
        }
        return true;
    }

    /**
     * get list of status names
     * @return array of strings
     */
    static function get_status_list(){
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";
        $query = "SELECT status_name FROM $table_name";
        $results = $wpdb->get_results($query,ARRAY_A);
        $status_list = array();
        foreach($results as $result){
            $status_list[] = $result['status_name'];
        }
        return $status_list;
    }

    /**
     * Gets the status name for an id
     * @param  [integer] $status_id [the status id to get its name]
     * @return [string/boolean]            [status name if found, false on failure]
     */
    static function get_status_name_for_id($status_id){
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";
        $query = "SELECT status_name FROM $table_name WHERE id = $status_id";
        $result = $wpdb->get_row($query,ARRAY_A);
        if($result){
            return $result['status_name'];
        }
        return false;
    }

    /**
     * Get the configured status color
     * @return array of status slug => color code
     */
    static function get_status_color_mapping(){
        global $wpdb;
        $table_name = $wpdb->prefix . "woocommerce_order_status_action";
        $query = "SELECT status_name,status_color FROM $table_name";
        $results = $wpdb->get_results($query,ARRAY_A);
        $status_list = array();
        foreach($results as $result){
            $status_list[$result['status_name']] = $result['status_color'];
        }
        $status_list['pending'] = '#999';
        $status_list['failed'] = '#e6db55';
        $status_list['processing'] = '#73a724';
        $status_list['completed'] = '#21759b';
        $status_list['on-hold'] = '#de9e0c';
        $status_list['cancelled'] = '#d54e21';
        $status_list['refunded'] = '#ccc';
        return $status_list;
    }
}
?>